--****************************************************************************************************************--
--! @ EDA322_processor
--! @ The top-level design of the processor datapath. Implemented using structural VHDL and previously constructed components. 
--! @ Signals and instantiations are generally written from left to right on the datapath schematic.

--! Authors: 
--! John William Croft 
--! Alfred Kulldorf
--****************************************************************************************************************--
----------------------------------------------------- Libraries ----------------------------------------------------
LIBRARY ieee; 
USE ieee.std_logic_1164.all; 
--USE ieee.numeric_std.all;
USE work.alu_pkg.all;
----------------------------------------------------- Entity -------------------------------------------------------
entity EDA322_processor is
	Port (
		externalIn : 		in STD_LOGIC_VECTOR (7 downto 0); -- ?extIn? in Figure 1
		CLK : 				in STD_LOGIC;
		master_load_enable: in STD_LOGIC;
		ARESETN : 			in STD_LOGIC;
		pc2seg :			out STD_LOGIC_VECTOR (7 downto 0); -- PC
		instr2seg :			out STD_LOGIC_VECTOR (11 downto 0); -- Instruction register
		Addr2seg :			out STD_LOGIC_VECTOR (7 downto 0); -- Address register
		dMemOut2seg : 		out STD_LOGIC_VECTOR (7 downto 0); -- Data memory output
		aluOut2seg : 		out STD_LOGIC_VECTOR (7 downto 0); -- ALU output
		acc2seg : 			out STD_LOGIC_VECTOR (7 downto 0); -- Accumulator
		flag2seg :		 	out STD_LOGIC_VECTOR (3 downto 0); -- Flags
		busOut2seg : 		out STD_LOGIC_VECTOR (7 downto 0); -- Value on the bus
		disp2seg: 			out STD_LOGIC_VECTOR (7 downto 0); --Display register
		errSig2seg :		out STD_LOGIC; -- Bus Error signal
		ovf : 				out STD_LOGIC; -- Overflow
		zero :				out STD_LOGIC); -- Zero				 
end EDA322_processor;
----------------------------------------------------- Architecture ------------------------------------------------- 
architecture dataflow of EDA322_processor is
	-------------- CONTROL OUTPUTS --------------------
	signal pcSel	: STD_LOGIC;
	signal pcLd 	: STD_LOGIC;
	signal instrLd 	: STD_LOGIC;
	signal im2bus 	: STD_LOGIC;
	signal addrMd	: STD_LOGIC;
	signal dmWr		: STD_LOGIC;
	signal dmRd 	: STD_LOGIC;
	signal dataLd	: STD_LOGIC;
	signal aluMd 	: STD_LOGIC_VECTOR(1 downto 0);
	signal accSel	: STD_LOGIC;
	signal accLd 	: STD_LOGIC;
	signal dispLd	: STD_LOGIC;
	signal acc2bus 	: STD_LOGIC;
	signal flagLd	: STD_LOGIC;
	signal ext2bus 	: STD_LOGIC;
	
	-------------- Bus ---------------
	signal BusOut 				: STD_LOGIC_VECTOR(7 downto 0);
	signal MemDataOutReged 		: STD_LOGIC_VECTOR(7 downto 0);
	signal OutFromAcc 			: STD_LOGIC_VECTOR(7 downto 0);
	
	-------------- Interconnect ---------------
	signal PCIncOut		: STD_LOGIC_VECTOR(7 downto 0);
	signal nxtpc 		: STD_LOGIC_VECTOR(7 downto 0);
	signal pc	 		: STD_LOGIC_VECTOR(7 downto 0);
	signal InstrMemOut	: STD_LOGIC_VECTOR(11 downto 0);
    signal Instruction	: STD_LOGIC_VECTOR(11 downto 0);
	signal Addr 		: STD_LOGIC_VECTOR(7 downto 0); -- Address into Data Mem.
	signal dataIn		: STD_LOGIC_VECTOR(7 downto 0);
	signal MemDataOut 	: STD_LOGIC_VECTOR(7 downto 0);
	signal ALU_out 		: STD_LOGIC_VECTOR(7 downto 0);
	signal FlagInp		: STD_LOGIC_VECTOR(3 downto 0); -- (MSB-LSB) => Ovf, NEQ, EQ, Zero
	signal ACC_in 		: STD_LOGIC_VECTOR(7 downto 0);
	signal OutFromFReg 	: STD_LOGIC_VECTOR(3 downto 0);
	
	-------------- CONTROL INPUTS --------------------
	alias opcode	: STD_LOGIC_VECTOR(3 downto 0) is Instruction(11 downto 8);
	alias NEQ	 	: STD_LOGIC is OutFromFReg(2);
	alias EQ	 	: STD_LOGIC is OutFromFReg(1);
	alias addrFromInstruction 	: STD_LOGIC_VECTOR(7 downto 0) is Instruction(7 downto 0);
	
begin
--------------------------------------------------------------------------------------------------------------------
-- Architecture: - Top Level
--------------------------------------------------------------------------------------------------------------------
	busOut2seg <= BusOut;
	dMemOut2seg <= MemDataOutReged;
	ovf <= FlagInp(3);
	zero <= FlagInp(0);
	
	controller : procController
		port map(	master_load_enable,
					opcode,
					NEQ,
					EQ,
					CLK,
					ARESETN,
					pcSel,
					pcLd,
					instrLd,
					addrMd,
					dmWr,
					dataLd,
					flagLd,
					accSEL,
					accLd,
					im2bus,
					dmRd,
					acc2bus,
					ext2bus,
					dispLd,
					aluMd);
					
	-------------- 7-Seg Out --------------------
	pc2seg 		<= pc;
	instr2seg	<= Instruction;
	Addr2seg 	<= Addr;
	dMemOut2seg	<= MemDataOutReged;
	aluOut2seg 	<= ALU_out;
	acc2seg		<= OutFromAcc;
	-- disp2seg directly connected in instance.
	flag2seg 	<= OutFromFReg;
	
	-------------- PC Increment --------------------
	PCInc : CLA_from_book
		--generic map(N => 8)
		port map(	S => PCIncOut,
					A => x"01",  -- constant increment by 1.
					B => pc,
					Cin => OPEN,
					Cout => OPEN); 
			
	-------------- INSTRUCTION MEMORY --------------------
	Instr_mem : mem_array 
		generic map(DATA_WIDTH 	=> 12,
					ADDR_WIDTH 	=> 8,
					INIT_FILE 	=> "inst_mem.mif")
		Port map(	ADDR 	=> pc,
					DATAIN 	=> x"000", -- Write Enable never active, so data irrelevant.
					CLK 	=> CLK,
					WE 		=> '0',
					OUTPUT 	=> InstrMemOut);
			
	
		
	-------------- DATA MEMORY --------------------
	dataIn <= BusOut;
	Data_mem : mem_array 
		generic map(DATA_WIDTH 	=> 8,
					ADDR_WIDTH 	=> 8,
					INIT_FILE 	=> "data_mem.mif")
		Port map(	ADDR 	=> Addr,
					DATAIN 	=> dataIn,
					CLK 	=> CLK,
					WE 		=> dmWr,
					OUTPUT 	=> MemDataOut);
	
	
	----------------------------------------------------- REGISTERS -------------------------------------------------------
	-------------- FE Register --------------------
	FE : D_register 
		--generic map(N => 8) 
		port map(	D 		=> nxtpc, 
					Aresetn => ARESETN, 
					Clock 	=> CLK, 
					Enable 	=> pcLd, 
					Q 		=> pc); 
	-------------- FE/DE Register --------------------
    FE_DE : D_register 
		generic map(N => 12) 
		port map(	D		=> InstrMemOut,
					Aresetn => ARESETN,
					Clock 	=> CLK,
					Enable 	=> instrLd, 
					Q  		=> Instruction);
	-------------- DE/EX --------------------
	DE_EX : D_register 
		--generic map(N => 8) 
		port map(	D 		=> MemDataOut, 
					Aresetn => ARESETN, 
					Clock 	=> CLK, 
					Enable 	=> dataLd, 
					Q 		=> MemDataOutReged);
					
	-------------- ACC  --------------------
	ACC : D_register 
		--generic map(N => 8) 
		port map(	D => ACC_in, 
					Aresetn => ARESETN, 
					Clock => CLK, 
					Enable => accLd, 
					Q => OutFromAcc);
					
	-------------- FReg --------------------
	FReg : D_register 
		generic map(N => 4) 
		port map(	D => FlagInp, 
					Aresetn => ARESETN, 
					Clock => CLK, 
					Enable => flagLd, 
					Q => OutFromFReg);
					
	-------------- ALU --------------------
	ALU : alu_wRCA 
		port map( 	ALU_inA => OutFromAcc, 
					ALU_inB => BusOut, 
					ALU_out => ALU_out, 
					Carry 	=> FlagInp(3), 
					NotEq 	=> FlagInp(2), 
					Eq 		=> FlagInp(1), 
					isOutZero => FlagInp(0), 
					operation => aluMd );
	
	-------------- (ACC) Display Register --------------------
	Display : D_register 
		--generic map(N => 8) 
		Port MAP(	D 		=> OutFromAcc, 
					Aresetn => ARESETN, 
					Clock 	=> CLK, 
					Enable 	=> dispLd,
					Q 		=> disp2seg); 

	-------------- MUX --------------------
	MUX_pcSel : mux21 port map( w0 => PCIncOut, w1 => BusOut, s => pcSel, 	f => nxtpc  ); 
	MUX_accSel: mux21 port map( w0 => ALU_out, 	w1 => BusOut, s => accSEL, 	f => ACC_in );
	MUX_addrMd:	mux21 port map( w0 => addrFromInstruction, w1 => MemDataOutReged, s => addrMd, f => Addr );
	
	-------------- DATA BUSS --------------------
    buss: procBus	
		Port map( 	INSTRUCTION => 	addrFromInstruction,
					DATA 		=> 	MemDataOutReged,
					ACC 		=>	OutFromAcc,
					EXTDATA 	=> 	externalIn,
					OUTPUT 		=> 	BusOut,
					ERR 		=> 	errSig2seg,
					instrSEL 	=> 	im2bus,
					dataSEL		=>  dmRd	,
					accSEL 		=> 	acc2bus,
					extdataSEL 	=> 	ext2bus);
end architecture;

