--****************************************************************************************************************--
--! @ procController
--! @ The main controller of the processor which generates control signals used in other modules.
--! @ The control logic is implemented using an FSM (Finite State Machine). 
--
--! Authors: 
--! John William Croft 
--! Alfred Kulldorff
--****************************************************************************************************************--
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
--use work.alu_pkg.all;


entity procController is
    port( 	
			master_load_enable 	: in 	std_logic;
			opcode 				: in  	std_logic_vector (3 downto 0);
			neq 				: in 	std_logic;
			eq 					: in 	std_logic; 
			CLK 				: in 	std_logic;
			ARESETN 			: in 	std_logic;
			pcSel 				: out  	std_logic;
			pcLd 				: out  	std_logic;
			instrLd 			: out  	std_logic;
			addrMd 				: out  	std_logic;
			dmWr 				: out  	std_logic;
			dataLd 				: out  	std_logic;
			flagLd 				: out  	std_logic;
			accSel 				: out  	std_logic;
			accLd 				: out  	std_logic;
			im2bus 				: out  	std_logic;
			dmRd 				: out  	std_logic;
			acc2bus 			: out  	std_logic;
			ext2bus 			: out  	std_logic;
			dispLd				: out 	std_logic;
			aluMd 				: out 	std_logic_vector(1 downto 0)
		);
end procController;

architecture Behavioral of procController is
	-- 5 states in Mealy machine.
	type STATE_TYPE is (FE, DE, DE2, EX, ME);
	signal curr_state, next_state : STATE_TYPE;
	
	-- Aggregate signal for all outputs.
	-- Indexed using constants below.
	signal ctrl_out : std_logic_vector(15 downto 0);
	
	-- Set specific control outputs in aggregate signal 'ctrl_out' using format: ctrl_out <= (sig1|sigN|sigM => '1', others => '0'); 
	-- This lets us use a name for a specific index position.
	constant   aluMd_low_c 	: integer :=0 ;
	constant   aluMd_high_c : integer :=1 ;
	constant   dispLd_c 	: integer :=2 ;
	constant   ext2bus_c 	: integer :=3 ;
	constant   acc2bus_c 	: integer :=4 ;
	constant   dmRd_c 		: integer :=5 ;
	constant   im2bus_c 	: integer :=6 ;
	constant   accLd_c 		: integer :=7 ;
	constant   accSel_c 	: integer :=8 ;
	constant   flagLd_c 	: integer :=9 ;
	constant   dataLd_c 	: integer :=10;
	constant   dmWr_c 		: integer :=11;
	constant   addrMd_c 	: integer :=12;
	constant   instrLd_c 	: integer :=13;
	constant   pcLd_c 		: integer :=14;
	constant   pcSel_c 		: integer :=15;
	
	-- OPCODE constants, useful in next state logic, though precludes manual minimization (minimization will be synthesized anyway).
	constant NOOP    : std_logic_vector(3 downto 0) := "0000";
	constant AD_ACC  : std_logic_vector(3 downto 0) := "0001";
	constant SU_ACC  : std_logic_vector(3 downto 0) := "0010";
	constant AND_ACC : std_logic_vector(3 downto 0) := "0011";
	constant NT_ACC  : std_logic_vector(3 downto 0) := "0100";
	constant CMP_ACC : std_logic_vector(3 downto 0) := "0101";
	constant LB_ACC  : std_logic_vector(3 downto 0) := "0110";
	constant SB_DM   : std_logic_vector(3 downto 0) := "0111";
	constant ADX_ACC : std_logic_vector(3 downto 0) := "1000";
	constant LBX_ACC : std_logic_vector(3 downto 0) := "1001";
	constant SBX_DM  : std_logic_vector(3 downto 0) := "1010";
	constant IN_DM   : std_logic_vector(3 downto 0) := "1011";
	constant J       : std_logic_vector(3 downto 0) := "1100";
	constant JNE     : std_logic_vector(3 downto 0) := "1101";
	constant JEQ     : std_logic_vector(3 downto 0) := "1110";
	constant DS      : std_logic_vector(3 downto 0) := "1111";
	
begin
	-- is there no way to concatenate outputs into a bus!!?
	-- ctrl_out <= pcSel & pcLd & instrLd & addrMd & dmWr & dataLd & flagLd & accSel & accLd & im2bus & dmRd & acc2bus & ext2bus & dispLd & aluMd;

	pcSel 	<= ctrl_out(pcSel_c) AND master_load_enable;
	pcLd 	<= ctrl_out(pcLd_c) AND master_load_enable;
	instrLd <= ctrl_out(instrLd_c);
	addrMd 	<= ctrl_out(addrMd_c);
	dmWr 	<= ctrl_out(dmWr_c);
	dataLd 	<= ctrl_out(dataLd_c);
	flagLd 	<= ctrl_out(flagLd_c);
	accSel 	<= ctrl_out(accSel_c);
	accLd 	<= ctrl_out(accLd_c);
	im2bus 	<= ctrl_out(im2bus_c);
	dmRd 	<= ctrl_out(dmRd_c);
	acc2bus <= ctrl_out(acc2bus_c);
	ext2bus <= ctrl_out(ext2bus_c);
	dispLd	<= ctrl_out(dispLd_c);
	aluMd 	<= ctrl_out(aluMd_high_c) & ctrl_out(aluMd_low_c);
	
	--------------------------------------------------------------------------------------------------------------------
	-- Process: sync_proc
	-- Description: The synchronous (register) part of the FSM. RESETs the current state and all outputs. Changes state on rising_edge when the master_load_enable is set, allowing the user to 'freeze' the current state.

	-- Input(s)	: ARESETN, master_load_enable, CLK, next_state
	-- Output(s): curr_state, ctrl_out
	--------------------------------------------------------------------------------------------------------------------
	sync_proc: process(ARESETN, CLK, master_load_enable)
	begin	
		if ARESETN = '0' then 
			curr_state <= FE;			-- Initialize STATE to FE, clear control output signals.
			--ctrl_out <= (others => '0');
		elsif master_load_enable='1' then
			if rising_edge(CLK) then     -- Rising edge infers FF, not latch!
				curr_state <= next_state;
			end if;
		end if;
	end process;
		
	--------------------------------------------------------------------------------------------------------------------
	-- Process: comb_next_state_proc
	-- Description: Concurrently computes the next state in the FSM using the current state and opcode input.

	-- Input(s)	: curr_state, opcode
	-- Output(s): next_state
	--------------------------------------------------------------------------------------------------------------------			
	comb_next_state_proc : process(curr_state, opcode)
	begin
		case(curr_state) is
			when FE =>
				next_state <= DE;  -- unconditional transition.
			when DE =>
				case opcode is
					when J | JNE | JEQ| IN_DM =>
						next_state <= FE;
					when ADX_ACC | LBX_ACC | SBX_DM =>
						next_state <= DE2;
					when NOOP | AD_ACC | SU_ACC | AND_ACC | NT_ACC | CMP_ACC | LB_ACC | DS =>
						next_state <= EX;
					when SB_DM =>
						next_state <= ME;
					when others => next_state <= FE; -- exception, return to initial state.
				end case;
			when DE2 =>
				case opcode is
					when ADX_ACC | LBX_ACC =>
						next_state <= EX;
					when SBX_DM =>
						next_state <= ME;
					when others => next_state <= FE; -- exception, return to initial state.
				end case;
			when ME =>
				next_state <= FE;
			when EX => 
				next_state <= FE;
			when others => next_state <= FE; -- exception, return to initial state.				
		end case;
	end process;
			
	--------------------------------------------------------------------------------------------------------------------
	-- Process: comb_output_proc
	-- Description: Concurrently computes the output of the FSM using the current state and opcode input.

	-- Input(s)	: curr_state, neq, eq, opcode
	-- Output(s): ctrl_out
	--------------------------------------------------------------------------------------------------------------------
	comb_output_proc : process (curr_state, neq, eq, opcode)
	begin
		case curr_state is
			------------------------ FE ----------------------------------
			when FE =>
				ctrl_out <= (instrLd_c => '1', others => '0'); -- instrLd
			------------------------ DE ----------------------------------
			when DE =>
				case opcode is
					when J | JNE | JEQ | IN_DM => -- DE->FE
						if opcode = IN_DM then 
							ctrl_out <= (pcLd_c | dmWr_c | ext2bus_c => '1', others => '0');
						elsif ((opcode = J) or (opcode = JEQ and eq = '1') or (opcode = JNE and neq = '1')) then -- if JUMP eval TRUE
							ctrl_out <= (pcLd_c | pcSel_c | im2bus_c => '1', others => '0');
						else 
							ctrl_out <= (pcLd_c | im2bus_c => '1', others => '0'); -- if JUMP eval FALSE, despite no jump im2bus is set so that bus in defined state.
						end if;	 
					when NOOP | AD_ACC | SU_ACC | AND_ACC | NT_ACC | CMP_ACC | LB_ACC | ADX_ACC | LBX_ACC | SBX_DM =>  
						ctrl_out <= (dataLd_c => '1', others => '0');
					when SB_DM => -- DE->ME
						ctrl_out <=(acc2bus_c => '1', others => '0');
					-- when DS => -- Slightly EVIL. Since it tells the synth tool that ctrl_out holds its value, latches are inferred.
						-- do nothing
					when others => ctrl_out <= (others => '0'); -- exception, reset all outputs.
				end case;
			------------------------ DE2 ----------------------------------
			when DE2 =>
				case opcode is
					when ADX_ACC | LBX_ACC =>
						ctrl_out <= (dataLd_c | addrMd_c => '1', others => '0');
					when SBX_DM =>
						ctrl_out <= (acc2bus_c => '1', others => '0');
					when others => ctrl_out <= (others => '0'); -- exception, reset all outputs.
				end case;
			------------------------ ME ----------------------------------
			when ME =>
				ctrl_out <= (pcLd_c | dmWr_c | acc2bus_c => '1', others => '0');
				if opcode = SBX_DM then
					ctrl_out(addrMd_c) <= '1';
				end if;  -- perhaps check for opcode validity here (others etc...) 
					
			------------------------ EX ----------------------------------
			when EX =>
				case opcode is 
					when NOOP | AD_ACC | ADX_ACC => 
						ctrl_out <= (pcLd_c | flagLd_c | accLd_c | dmRd_c => '1', others => '0'); -- aulMd = "00"
					when SU_ACC =>
						ctrl_out <= (pcLd_c | flagLd_c | accLd_c | dmRd_c | aluMd_low_c => '1', others => '0');
					when AND_ACC =>
						ctrl_out <= (pcLd_c | flagLd_c | accLd_c | dmRd_c | aluMd_high_c => '1', others => '0');
					when NT_ACC =>
						ctrl_out <= (pcLd_c | flagLd_c | accLd_c | dmRd_c | aluMd_high_c | aluMd_low_c => '1', others => '0');
					when CMP_ACC =>
						ctrl_out <= (pcLd_c | flagLd_c | dmRd_c => '1', others => '0');
					when LB_ACC | LBX_ACC =>
						ctrl_out <= (pcLd_c | accLd_c | accSel_c | dmRd_c => '1', others => '0');
					when DS => 
						ctrl_out <= (pcLd_c |  dispLd_c => '1', others => '0'); -- aluMd = "00"
					when others => ctrl_out <= (others => '0'); -- exception, reset all outputs
				end case;
			when others => ctrl_out <= (others => '0'); -- exception, reset all outputs.
		end case;
	end process;
			
			
end Behavioral;


