/* LIBRARIES */
#include "TinyTimber.h"
#include "sciTinyTimber.h"
#include "canTinyTimber.h"
#include "sioTinyTimber.h"
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h> // For TRUE/FALSE statements.
#include <string.h>  // String manipulation.


//=========//
/* DEFINES */
//=========//
#define MAX_VOL 25

#define MAX_IDX 14
#define MIN_IDX -10

#define MIN_KEY -5
#define MAX_KEY  5

#define MIN_TEMPO 60 // bpm
#define MAX_TEMPO 240

// typedef volatile unsigned int* port16ptr;
// #define DAC_ADDR 0x4000741C
// #define DACport *((port16ptr) DAC_ADDR)
#define DACport DAC->DHR8R2

// Modes of operation for the USER button.
#define MOMENTARY 0
#define HOLD      1

#define FIFO_SIZE 11 // fifo capacity is FIFO_SIZE - 1
#define FIFO_FULL -1
#define FIFO_EMPTY -1

#define CAN_DELTA SEC(1)

// Debug printouts
#define BUTTON_DEBUG 0
#define CAN_DEBUG 1    // Verbosity levels 0: none, 1: deliveries only, 2: all


// Whole note, double note, half note.
typedef enum {a, b, c} beat_t;

// Indices originally have origin (ie. pos. 0) at 440Hz. Shifted by MIN_IDX to make freqs vector addressable.
// BRODER JAKOB
int period_idx[] =  {10,12,14,10,10,12,14,10,14,15,17,14,15,17,17,19,17,15,14,10,17,19,17,15,14,10,10,5,10,10,5,10}; 				// Indexed by seq_idx. THE SONG TO PLAY
beat_t beat_arr[] = { a, a, a, a, a, a, a, a, a, a, b, a, a, b, c, c, c, c, a, a, c, c, c, c, a, a, a, a, b, a, a, b }; 			// indexed by seq_idx

//// TO ZANARKAND
//int period_idx[] =  {17,5,8,12,17,19,20, 15,3,7,10,15,17,19};
//beat_t beat_arr[] = { a,a,a,a ,a ,a ,b,   a,a,a,a ,a ,a ,b };

int period[] = {2024,1911,1803,1702,1607,1516,1431,1351,1275,1203,1136,1072,1012,955,901,851,803,758,715,675,637,601,568,536,506}; 	// indexed by idx
int idxidx_arr[] = {4,8,11,14,20,26,29}; // CAN CANON beat changes?

//============================//
// Class declarations	  	  //
//============================//
typedef struct {
	// Misc. + serial input parsing.
    Object super;
    int count;
    char c;
	char buff[100]; 	// Keyboard input buffer.
	char strBuff[100];	// For debug printouts.
	// 1. Befriend that Button
	Time lastBaseline; 
	Time now;
	Time diff;
	bool buttonPressed;
	int userMode;
	int pressID;
	// 2. Rule that Regulator
	int canMsgID;	// Range (0-127), post-incremented.	
	bool debugOutput;
	Time canLastDelivery;
	int fifo[FIFO_SIZE]; // Implemented as circular buffer using read/write pointers (see below).
	int fifoWritePtr;
	int fifoReadPtr;
} App;
App app = { initObject(), 0, 'X', {0}, {0}, 0, 0, 0, false, MOMENTARY, 0, 0, false, 0, {0}, 0, 0};

typedef struct {
	Object super;
	int volume;
	int period;    // usec
	int deadline; // usec
	bool last;
	int lastVolume;
	bool mute;
   bool alive;
} ToneGen;
ToneGen t1 = { initObject(), 25, 500, 100, false, 0, false, true };

typedef struct {
	Object super;
	int seq_idx;
	int temp_tempo; // save received tempo until a "transition note" ie. 4,8,11,14,20,26,29 in seq_idx
	int tempo;  // beats per minute
	int key;
	bool silence;
   Time noteLength;
   bool alive;
   bool terminated;
   Msg ID;
} Sequencer;
Sequencer s1 = { initObject(), 0, 120, 120, 0, false, 1, false, true }; // Msg ID omitted.

//============================//
//      Function prototypes   //
//============================//
// SERIAL stuff
void reader(App*, int);

// CAN stuff
void receiver	(App*, int); 	// CAN receiver.
void transmitter(App*, int);    // CAN transmitter. 
void canBurstMode(App* , int);   // Periodic transmitter.
int  fifoWrite   (App*, int);
int  fifoRead    (App*, int);

// SIO stuff
void buttonTrig(App*, int);

// MISC stuff
void reset(App*, int); // Reset all state variables.

void incVol(ToneGen*, int);
void decVol(ToneGen*, int);
void muteVol(ToneGen*, int);

void setPeriod(ToneGen*, int);
void enable(ToneGen*, int);
void kill(ToneGen*, int);

void SEQUENCE	(Sequencer*, int);
void PLAY		(Sequencer*, int);
void setTempo	(Sequencer*, int);
void setKey		(Sequencer*, int);
void setPos		(Sequencer*, int);
int readTempo	(Sequencer*, int);

//============================//
//      Object INSTANTIATIONS //
//      & INIT           	  //
//============================//
Serial sci0 = initSerial(SCI_PORT0, &app, reader);
Can    can0 = initCan   (CAN_PORT0, &app, receiver);
SysIO  sio0 = initSysIO (SIO_PORT0, &app, buttonTrig);

void reset(App* self, int unused){
	// TODO
	// app = { initObject(), 0, 'X', {0}, {0}, 0, 0, 0, false, MOMENTARY, 0, 0, false, 0, {0}, 0, 0};
}

int fifoWrite(App* self, int elem){
	if ( ((self->fifoWritePtr + 1) % FIFO_SIZE) == self->fifoReadPtr ){	// Simplest way to disambiguate empty vs. full fifo.
		return FIFO_FULL;												// In this case, full fifo is defined as being one incrementation away from read/write pointers matching.	
	}
	else{
		self->fifo[self->fifoWritePtr] = elem;						// Write element to buffer.
		self->fifoWritePtr = (self->fifoWritePtr + 1) % FIFO_SIZE;	// Increment write pointer.
	}
	return 0;
}

int fifoRead(App* self, int unused){
	int ret;
	if (self->fifoReadPtr == self->fifoWritePtr){	// If write/read pointers match, buffer is defined as empty.
		return FIFO_EMPTY;
	}
	else{
		ret = self->fifo[self->fifoReadPtr];
		self->fifoReadPtr = (self->fifoReadPtr + 1) % FIFO_SIZE;
		
	}
	return ret;
}

// Get the relative position of the write pointer. ie. the distance between write/read pointers.
int fifoGetRelPos(App* self, int unused){
	return (FIFO_SIZE + (self->fifoWritePtr - self->fifoReadPtr)) % FIFO_SIZE;
}

void canBurstMode(App *self, int unused){
	if (self->userMode == HOLD){
		transmitter(self, 0);
		AFTER(MSEC(500), self, canBurstMode, 0);
	}
}

void changeUserMode(App *self, int calledPressID){
	if (self->pressID == calledPressID){         // If the ID passed to the function = the current ID in the state variable, the button has not been released yet.
		self->userMode = HOLD;
		if (BUTTON_DEBUG){
			SCI_WRITE(&sci0, "HOLD-MODE on\n");
			snprintf(self->strBuff, 100, "Time to HOLD mode (1s): %03i%03i ms\n", SEC_OF(CURRENT_BASELINE()-self->lastBaseline), MSEC_OF(CURRENT_BASELINE()-self->lastBaseline)); 
			SCI_WRITE(&sci0, self->strBuff);
		}
		SIO_WRITE(&sio0, 0); // Switch on LED.
		
		canBurstMode(self, 0);
	}
}

void buttonTrig(App *self, int unused) {
	
	if (self->buttonPressed == false){
		self->buttonPressed = true; 		// The button is being held down.
		SIO_TRIG(&sio0, 1); 				// Trigger next on button press->release.
		self->now = CURRENT_BASELINE();         		// Store arrival time.
		self->diff = self->now - self->lastBaseline;	// Time difference between subsequent presses.
		
		if (BUTTON_DEBUG){
			snprintf(self->strBuff, 100, "pressID: %i @ %i%03i ms\n", self->pressID, SEC_OF(self->now), MSEC_OF(self->now)); 
			SCI_WRITE(&sci0, self->strBuff);
			snprintf(self->strBuff, 100, "Button DELTA: %3i%03i ms\n", SEC_OF(self->diff), MSEC_OF(self->diff)); 
			//                                             ^- %03i : ensure value length of 3, and pad with '0' if necessary. Padding optional, blank spaces otherwise.
			SCI_WRITE(&sci0, self->strBuff);
		}
		
		if (self->diff < MSEC(100)){
			if (BUTTON_DEBUG)
				SCI_WRITE(&sci0, "Bounce detected.\n");
		}else{
			self->lastBaseline = self->now;	// Update time for last valid button press.
			AFTER(SEC(1), self, changeUserMode, self->pressID);	// Check if this unique button press NOT released >=1sec from now.
			
		}
	}
	else{
		self->buttonPressed = false;// The button is released.
		SIO_TRIG(&sio0, 0); 		// Trigger next on release->press.
		self->pressID++;  			// Increment unique button-push ID on release.
		
		if(self->userMode == HOLD){
			self->userMode = MOMENTARY;
			SIO_WRITE(&sio0, 1); // Switch off LED.
			if (BUTTON_DEBUG){
				snprintf(self->strBuff, 100, "HOLD-MODE off. Duration: %03i%03i ms\n", SEC_OF(CURRENT_BASELINE()-self->lastBaseline), MSEC_OF(CURRENT_BASELINE()-self->lastBaseline)); 
				SCI_WRITE(&sci0, self->strBuff);
			}
		}
		else{
			transmitter(self, 0); // Transmit CAN message on release.
		}
		if (BUTTON_DEBUG){
			SCI_WRITE(&sci0, "\n"); // Always end a button press with a newline for pretty output.
		}
	}
}

void transmitter(App* self, int unused){
	Time now = CURRENT_BASELINE();
	
	CANMsg msg;
	msg.msgId = self->canMsgID;
    msg.nodeId = 1;
    msg.length = 6;
    msg.buff[0] = 'H';	// Actual message NOT important.
    msg.buff[1] = 'e';
    msg.buff[2] = 'l';
    msg.buff[3] = 'l';
    msg.buff[4] = 'o';
    msg.buff[5] = 0;
    CAN_SEND(&can0, &msg);
	
	self->canMsgID = (self->canMsgID + 1) % 128;	// Range [0-127] then wraps around.
	
	if (self->debugOutput){
		snprintf(self->strBuff, 100, "%3i%03i ms: CAN msg transmit. msgid: %3i\n", SEC_OF(now), MSEC_OF(now), msg.msgId); 
		SCI_WRITE(&sci0, self->strBuff);
	}
}

void deliverFromFIFO(App *self, int unused){
	Time now = CURRENT_BASELINE();
	int msgId = fifoRead(self, 0);				// Deliver msg at top of buffer.
	self->canLastDelivery = now;	// Update last delivery time.
	
	if (msgId == FIFO_EMPTY){
		snprintf(self->strBuff, 100, "%3i%03i ms: ERROR! FIFO empty. VERY STRANGE! Check if too many deliveries are being scheduled.\n", SEC_OF(now), MSEC_OF(now)); 
		SCI_WRITE(&sci0, self->strBuff);
	}
	else{
		if (CAN_DEBUG >= 1){
			snprintf(self->strBuff, 100, "%3i%03i ms: Delivery (from FIFO). msgid: %3i\n", SEC_OF(now), MSEC_OF(now), msgId); 
			SCI_WRITE(&sci0, self->strBuff);
		}
	}
}



void receiver(App *self, int unused) { //CAN Receiver
	
	CANMsg msg;
    CAN_RECEIVE(&can0, &msg);
	
	Time t_in = CURRENT_BASELINE();
	
	if (CAN_DEBUG >= 2){
		snprintf(self->strBuff, 100, "%3i%03i ms: Arrival. msgid: %3i\n", SEC_OF(t_in), MSEC_OF(t_in), msg.msgId);
		SCI_WRITE(&sci0, self->strBuff);
	}
	//// REGULATOR ////
	// Buffers incoming CAN messages and delivers them with a minimum inter-arrival time, CAN_DELTA.
	// Uses a circular FIFO.
	////
	if ( (t_in - self->canLastDelivery >= CAN_DELTA) && (self->fifoReadPtr == self->fifoWritePtr)) {	
			self->canLastDelivery = t_in;	// Update t_out.
			if (CAN_DEBUG >= 1){
				snprintf(self->strBuff, 100, "%3i%03i ms: Delivery (direct). msgid: %3i\n", SEC_OF(t_in), MSEC_OF(t_in), msg.msgId); 
				SCI_WRITE(&sci0, self->strBuff);
			}
	}
	else{
		int fifoStatus = fifoWrite(self, msg.msgId);	// Add message to buffer.
		
		if (fifoStatus == FIFO_FULL){
			snprintf(self->strBuff, 100, "%3i%03i ms: ERROR! fifo full, msgId: %3i discarded. Delivery not scheduled!\n", SEC_OF(t_in), MSEC_OF(t_in), msg.msgId);
			SCI_WRITE(&sci0, self->strBuff);
		}
		else{
			AFTER((self->canLastDelivery+CAN_DELTA-t_in) + CAN_DELTA*(fifoGetRelPos(self, 0)-1), self, deliverFromFIFO, 0);
			
			if (CAN_DEBUG >= 2){
				snprintf(self->strBuff, 100, "%3i%03i ms: CAN msg added to FIFO. msgid: %3i. Pos: %i\n", SEC_OF(t_in), MSEC_OF(t_in), msg.msgId, fifoGetRelPos(self, 0)); 
				SCI_WRITE(&sci0, self->strBuff);
			}
		}
	}
}










void reader(App *self, int c) { //Serial Reader
    SCI_WRITE(&sci0, "Rcv: \'");
    SCI_WRITECHAR(&sci0, c);
    SCI_WRITE(&sci0, "\'\n");
	
	/* Keyboard functions:
	 * 'p': play tune
	 * 'd': toggle CAN debug printouts
	 * '1': print CAN FIFO contents.
	 * 'r': reset state variables.
	 */
	switch((char)c){
		case ('p'):
			ASYNC(&s1, PLAY, 0);
			SCI_WRITE(&sci0, "PLAY toggled\n");
			break;
		case ('d'): // Debug messages toggle.
			self->debugOutput = !(self->debugOutput);
			if(self->debugOutput == true){
				SCI_WRITE(&sci0, "CAN debug printout enabled\n");
			}				
			else{
				SCI_WRITE(&sci0, "CAN debug printout disabled\n");
			}
			break;
		case('1'):
			SCI_WRITE(&sci0, "Buffer contents:\n");
			
			for(int i = 0; i < FIFO_SIZE; i++){
				snprintf(self->strBuff, 100, "elem %3i: %i", i, self->fifo[i]); 
				SCI_WRITE(&sci0, self->strBuff);
				if (self->fifoWritePtr == i)
					SCI_WRITE(&sci0, " <head");
				if (self->fifoReadPtr == i)
					SCI_WRITE(&sci0, " <tail");
				SCI_WRITE(&sci0, "\n");
			}
			break;
		case('r'):
			SCI_WRITE(&sci0, "TODO: Resetting!\n");
			reset(self, 0);
			break;
		default:
			break;
	}
}

//============================//
//      ToneGen METHODS       //
//============================//
void SQUARE(ToneGen *self, int unused){
   if(self->alive){
   
      if (self->last == true){
         DACport = 0x0;} // DAC output 2 / 8 bit right aligned
      else{
         DACport = self->volume;}

     // Update last state-aa
      self->last = !(self->last);

      SEND(USEC(self->period), USEC(self->deadline) ,self, SQUARE, 0);
   }
   else{ // If killed, set output low.
	   self->last = false;
	   DACport = 0x0;
   }
}

// Enable the tonegen at the beginning of each note. Kill the task to "generate" silence.
void kill(ToneGen *self, int unused){
   self->alive = false;
}

void enable(ToneGen *self, int unused){
   self->alive = true;
}

// Volume control
void incVol(ToneGen *self, int unused){
	SCI_WRITE(&sci0, "incVol\n");
  if (self->volume < MAX_VOL){
    self->volume++;
	self->mute = false;
  }
}

void decVol(ToneGen *self, int unused){
	SCI_WRITE(&sci0, "decVol\n");
  if (self->volume > 0){
    self->volume--;
	if (self->volume == 0){
		self->mute = true;
		self->lastVolume = 0;
	}
  }
}

void muteVol(ToneGen *self, int unused){
	SCI_WRITE(&sci0, "mutVol\n");
	
	if (self->volume != 0 && self->mute == false){
		self->lastVolume = self->volume;
		self->mute = true;
		self->volume = 0;
	}else{
		self->volume = self->lastVolume;
		self->mute 	 = false;
	}
}

void setPeriod(ToneGen* self, int new_period){
	self->period = new_period;
}

//============================//
//      Sequencer METHODS       //
//============================//

void PLAY(Sequencer* self, int unused){
	if(self->alive == true){ // PLAY->STOP
		self->alive = false;
		
	}else if(self->terminated == true && self->alive == false){ // STOP -> PLAY
		self->seq_idx = 0;
		self->ID = ASYNC(&s1, SEQUENCE, 0); 
		self->alive = true;
	}
}

void setPos(Sequencer* self, int pos){
		self->seq_idx = pos;
}

void setKey(Sequencer* self, int newKey){
   if(newKey <= MAX_KEY && newKey >= MIN_KEY){
      self->key = newKey;
      SCI_WRITE(&sci0, "Key changed LOCAL.");
   }
   else
      SCI_WRITE(&sci0, "Out of bounds!");
}

void setTempo(Sequencer* self, int newTempo){
	   if(newTempo <= MAX_TEMPO && newTempo >= MIN_TEMPO){
		  self->temp_tempo = newTempo;
		  SCI_WRITE(&sci0, "Tempo changed LOCAL.");
	   }
	   else
		  SCI_WRITE(&sci0, "Out of bounds!");
}

/* The sequencer can be viewed as a two-state FSM: tone and silence states.
// The states will alternate and the actions on each transition are given by the if-else statement below.
// Tone->Silence
   // Kill the squarewave generator (ie. stop it from invoking itself).
   // Increment sequence position (seq_idx).
   // Schedule next Sequencer transition using silence length.
// Silence->Tone
   // Calculate total note length from TEMPO and current BEAT.
   // Send current tone freq. to ToneGen (setPeriod())
   // Enable the squarewave generator.
   // "Kickoff" the squarewave generator ie. call it initially, it will call itself recursively therafter.
   // Schedule next Sequencer transition using the calculated note length (and a silence length [which may be unknown]).
*/
void SEQUENCE(Sequencer* self, int unused){
		self->terminated = false;
		
		if(self->silence == false){
			
			if(	self->seq_idx == 0 ||
				self->seq_idx == 4 ||
				self->seq_idx == 8 ||
				self->seq_idx == 11 ||
				self->seq_idx == 14 ||
				self->seq_idx == 20 ||
				self->seq_idx == 26 ||
				self->seq_idx == 29)
					{
						self->tempo = self->temp_tempo; // update the ACTUAL tempo.
					}
						
		
			self->noteLength = MSEC( 1000*60 / (self->tempo) ); // gives usec in type Time from bpm
			
			/*
			char debug_buff[100];
			snprintf(debug_buff, 100, "%d", self->seq_idx);
			SCI_WRITE(&sci0, debug_buff);
			*/
			
			/*
			snprintf(debug_buff, 100, "%d", t1.volume);
			SCI_WRITE(&sci0,"ToneGen Volume ");
			SCI_WRITE(&sci0, debug_buff);
			SCI_WRITE(&sci0,"\n");
			
			snprintf(debug_buff, 100, "%u", MSEC_OF(self->noteLength));
			SCI_WRITE(&sci0,"Initial noteLen before beat modulation: ");
			SCI_WRITE(&sci0, debug_buff);
			SCI_WRITE(&sci0,"\n\n");
			 * */
		  
		  switch(beat_arr[self->seq_idx]){
			 case(a):
				;
			 break;
			 case(b):
				self->noteLength *= 2;
			 break;
			 case(c):
				// self->noteLength /= 2;
				self->noteLength = self->noteLength >> 1;
			 break;
			 default:
			 break;
		  }
		  /*
		  snprintf(debug_buff, 100, "%lu", self->noteLength);
			SCI_WRITE(&sci0,"Initial noteLen AFTER beat modulation: ");
			SCI_WRITE(&sci0, debug_buff);
			SCI_WRITE(&sci0,"\n\n");
		   
			int current_period = period[period_idx[self->seq_idx] + self->key];
			
			snprintf(debug_buff, 100, "%u", current_period);
			SCI_WRITE(&sci0,"Current period (sent to tonegen) ");
			SCI_WRITE(&sci0, debug_buff);
			SCI_WRITE(&sci0,"\n");
			*/
			
			int current_period = period[period_idx[self->seq_idx] + self->key];
			
			 ASYNC(&t1, setPeriod, current_period);

			 ASYNC(&t1, enable, 0);
			 ASYNC(&t1, SQUARE, 0);
			
			SEND(self->noteLength - MSEC(50) ,0,self, SEQUENCE, 0);
			self->silence = true;
		  
		}
		else{		
			ASYNC( &t1, kill, 0); // 
			
			// SCI_WRITE(&sci0,"Silence state\n");
			
		  
		  if(self->seq_idx++ == sizeof(period_idx)/sizeof(int) - 1 ) // Increment position in melody. If reached end, wrap to 0. 
			 self->seq_idx = 0;

		self->silence = false;
		
		if(self->alive == true){
			self->ID = SEND(MSEC(50) ,0,self, SEQUENCE, 0);
		}else{
			self->terminated = true;
		}
	}
}

int readTempo(Sequencer *self, int unused){
	return self->tempo;
}

/* ================================================================================================================================= */
void startApp(App *self, int arg) {
	// Driver init
    CAN_INIT(&can0);
    SCI_INIT(&sci0);
	
	SIO_INIT (&sio0);
	SIO_TRIG (&sio0, 0); // Which button transition to trig on. 0=on press, 1=on release.
	SIO_WRITE(&sio0, 1); // Initialize LED off. Inverted logic.
	
    SCI_WRITE(&sci0, "Hello, hello...\n");

	/*
	CANMsg msg;	
    msg.msgId = 1;
    msg.nodeId = 1;
    msg.length = 6;
    msg.buff[0] = 'H';
    msg.buff[1] = 'e';
    msg.buff[2] = 'l';
    msg.buff[3] = 'l';
    msg.buff[4] = 'o';
    msg.buff[5] = 0;
    CAN_SEND(&can0, &msg);
	 */
}

int main() {
    INSTALL(&sci0, sci_interrupt, SCI_IRQ0);
    INSTALL(&can0, can_interrupt, CAN_IRQ0);
	INSTALL(&sio0, sio_interrupt, SIO_IRQ0);
	
    TINYTIMBER(&app, startApp, 0);
}
